 /**
 *
 * \file        HPRFGW_HEARTBEAT.c
 *
 * \brief       This file contains the RF device heart beat task  	
 *
 * \detail		
 *
 * \author      Hazrat Shah	
 *
 * \date        10/10/2006
 * \warning	
 * \note
 * \review
 */

#include "hprfgw_config.h"
#include "hprfgw_rfApp.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfHeartbeat.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfSlaveIntToHost.h"

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS	
////////////////////////////////////////////////////////////////////////////////
#define	RF_HEART_BEAT_TASK_RATE			10000		//	10 seconds	
#ifdef	RF_NETWORK_MASTER_DEVICE
 #define RF_BEACON_RATE                  2000        //2 SECONDS
#endif

#define	RF_HEART_BEAT_TASK_STACK_SIZE	(8*1024)
#ifndef RF_TASKS_BASE_PRIORITY
#if defined (OS_NUCLEUS)
 #pragma info "!!!!Pass in the RF_TASKS_BASE_PRIORITY using the make file!!!!"
#endif
#endif
//TODO: Does dm_os.h have standard priority levels defined?
#define RF_HEART_BEAT_TASK_PRIORITY		RF_TASK_PRIORITY(RF_TASKS_BASE_PRIORITY, 2)

#define	RF_PHY_RX_LOCKUP_PATCH

////////////////////////////////////////////////////////////////////////////////
//	LOCAL DATA		
////////////////////////////////////////////////////////////////////////////////
typedef struct	{
#if defined (OS_NUCLEUS)
   	NU_TASK 	task;
	UINT8   	stack[RF_HEART_BEAT_TASK_STACK_SIZE];
#elif defined (OS_WINCE)
	HANDLE		task;
#endif
	DM_RESOURCE_HANDLE task;
}	TYPE_RF_HEART_BEAT_TASK;
TYPE_RF_HEART_BEAT_TASK RFHeartBeatTask;

////////////////////////////////////////////////////////////////////////////////
//	
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The heart beat task  
 *
 * \detail		This function checks the heart beat task of all devices whose link state is active.  
 *				If the device heart beat time expires the heart beat timed out flag and HeartBeat Timeout Event gets set. 
 *				The rf receiver task waits on the heart beat timeout event 
 *
 * \date        09/13/2006
 *
 * \param	    argc, argv			
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
#if defined (OS_NUCLEUS)
void	RFHeartBeat_Task (UINT32 argc, void * argv)
#elif defined (OS_WINCE)
DWORD RFHeartBeat_Task (void * pContext)
#endif
static void RFHeartBeat_Task(UINT32 param)
{
#ifdef	RF_NETWORK_MASTER_DEVICE
	UINT16	i;
	UINT8	AnyDeviceHeartBeatTimeoutOccuredFlag;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst;
#endif

#ifdef	RF_NETWORK_MASTER_DEVICE	
    #define HB_WAIT_CNT RF_HEART_BEAT_TASK_RATE/RF_BEACON_RATE
	INT8    HB_waitCnt = HB_WAIT_CNT;
	UINT32  randomNum = 0;
#endif

	RFIntHostApp.pF_StartupWait(HPRF_STARTUP_HEARTBEAT_READY);

	//	Wait for RF Stack to be intialized
	while (!RFReceive_IsRFStackInitialized())	{	HwDelayMsec(100);	}

#if defined (OS_WINCE)
	RFIntHostApp.pF_ErrorPrintf("\r\nRFHeartBeat_Task: Running!");
#endif

	while (1)	{

#ifdef	RF_NETWORK_MASTER_DEVICE
		if( --HB_waitCnt <= 0)
		{

			HB_waitCnt = HB_WAIT_CNT;

			pdst = g_pRFDeviceDescriptiveTable;
			AnyDeviceHeartBeatTimeoutOccuredFlag = FALSE;
	
			for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
				if (pdst->linkstate == RF_DEVICE_LINK_STATE_ACTIVE)	{
					if (pdst->HeartBeatTimer)	{
						if (--pdst->HeartBeatTimer == 0)	{
							pdst->HeartBeatTimedOut = TRUE;
							AnyDeviceHeartBeatTimeoutOccuredFlag = TRUE;	
						}	
					}
				}		
			}
	
			if (AnyDeviceHeartBeatTimeoutOccuredFlag)	{
				RFReceive_SetHeartBeatTimeoutEvent();	
			}

#ifdef	RF_PHY_RX_LOCKUP_PATCH	
			if(!g_roamingEnabled)//transmission beacon message will unlock the PHY
			{
				//	Note: If there are no devices attached, reset the rf stack
				//		  Once in a while the PHY locks-up and cannot receive packets
				//		  This mechanism is not needed in the slave devices, because non-broadcast  
				//		  transmission from the slave device will unlock the PHY
				BOOL	resetphy;
				UINT32	i;
				for (i=0, resetphy=TRUE; i<g_MaxSupportedRFDevices; i++)	{
					if (g_pRFDeviceDescriptiveTable[i].linkstate == RF_DEVICE_LINK_STATE_ACTIVE)	{
						resetphy = FALSE;
						break;
					}
				}
				if (resetphy)	{
					RFTransmit_BuildResetMACMessageAndPostToPipe(FALSE);
					gMStatp.NoDevConnSMACResetCntr++;				 
				}
			}
#endif
		}//end of if (--HB_waitCnt <= 0)

		if(g_roamingEnabled)
        {
			randomNum = rand();
			RFIntHostApp.pF_HwDelayMsec((randomNum%16)*30);//select one of the 16 30ms time slot
			RFAPP_PrepareAndSubmitRFBeaconPacket();
			//RFIntHostApp.pF_ConsolePrintf("randomNum%16=%d,num/2045[%d],num[%d]\r\n",randomNum%16,randomNum/2045,randomNum);
		}		
		RFIntHostApp.pF_HwDelayMsec(RF_BEACON_RATE);
#else
		//	Send heart beat to the master station every x-seconds
		BOOL isGatewayConnected = RFReceive_IsGatewayConnected();
		BOOL isAcquiring = RFAPP_AcquireStatus();
		HPRF_DEBUG(DmConsolePrintf("%s --- gateway connected = %i isAcquiring = %i\n", \
		        __FUNCTION__, isGatewayConnected, isAcquiring);)
		if(isGatewayConnected && !isAcquiring) {
			if (--g_pRFDeviceDescriptiveTable[DEVICE_DESCRIPTIVE_TABLE_INDEX].HeartBeatTimer == 0)	{
				//	Send heart beat packet to the master stattion			
				RFTransmit_BuildHeartBeatPacketAndPostToPipeByIndex(DEVICE_DESCRIPTIVE_TABLE_INDEX);
				RFHeartBeat_ReTriggerTimer (DEVICE_DESCRIPTIVE_TABLE_INDEX);
			}
		}
        HwDelayMsec (RF_HEART_BEAT_TASK_RATE);
#endif
	}
}

////////////////////////////////////////////////////////////////////////////////
//	GLOBAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reloads the rf device heart beat timer	 	   
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    index 		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFHeartBeat_ReTriggerTimer (UINT8 index)	{	
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
		
	pdst->HeartBeatTimedOut = FALSE;
	pdst->HeartBeatTimer 	= RFReceive_GetLocalDeviceHeartBeatTimeout();
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks all rf device heart beat timeout 	   
 *
 * \detail		If a heart beat or any other packet is not received from the RF device the device heart beat timer expires
 *				the device link state is then set to FAIL.  
 *
 * \date        10/11/2006
 *
 * \param	    None		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFHeartBeat_TimeoutProcessing (void)	{	
	UINT16	i;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;

	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if (pdst->HeartBeatTimedOut)	{
			pdst->HeartBeatTimedOut = FALSE;
			pdst->COMLog.heartbeattimeouts++;
			RFDesc_LinkStateTransition ((UINT8)i, RF_DEVICE_LINK_STATE_FAIL);
		}
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The rf heart beat initialization 
 *
 * \detail		This function creates the heart beat task	  
 *
 * \date        10/12/2006
 *
 * \param	    None		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFHeartBeat_Init (void)
{
    //	Create rf heart beat task	 
#if defined (OS_NUCLEUS)
    if(NU_Create_Task(&RFHeartBeatTask.task, "TRFHEARTB", RFHeartBeat_Task, 0, NU_NULL,
        			   RFHeartBeatTask.stack, sizeof(RFHeartBeatTask.stack), RF_HEART_BEAT_TASK_PRIORITY, 0, NU_PREEMPT, NU_START) != NU_SUCCESS)	{
        RFIntHostApp.pF_ErrorHandler("TRFHEARTB: Could not create task!");
    }
#elif defined (OS_WINCE)
	RFHeartBeatTask.task = CreateThread(NULL, 0, RFHeartBeat_Task, pHPRDevice, 0, NULL);
	if (NULL == RFHeartBeatTask.task)
	{
		RFIntHostApp.pF_ErrorPrintf("TRFHEARTB: Fail create task!");
	}
	// Set thread priority
	CeSetThreadPriority(RFHeartBeatTask.task, RF_HEART_BEAT_TASK_PRIORITY);
#endif
	RFHeartBeatTask.task = OsCreateNamedAdvTaskPri(RFHeartBeat_Task, 0, 0, RF_HEART_BEAT_TASK_PRIORITY, "TRFHEARTB");
	if (!RFHeartBeatTask.task)
	{
		RFIntHostApp.pF_ErrorPrintf("TRFHEARTB: Fail create task!");
	}
}


